/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <errno.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_proto.h"
#include "udm_services.h"
#include "udm_agent.h"
#include "udm_db.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_sdp.h"
#include "udm_xmalloc.h"
#include "udm_searchtool.h"
#include "udm_vars.h"
#include "udm_word.h"
#include "udm_db_int.h"


/*
#define DEBUG_SDP
*/

/*
#define DEBUG_SEARCH
*/

static ssize_t UdmSearchdSendPacket(int fd,const UDM_SEARCHD_PACKET_HEADER *hdr,const void *data){
	ssize_t nsent;
	
	nsent=UdmSend(fd,hdr,sizeof(*hdr),0);
	if(nsent!=sizeof(*hdr))
		return(nsent);
	if(data){
		nsent+=UdmSend(fd,data,hdr->len,0);
	}
	return nsent;
}


#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

static int open_host(char *hostname,int port, int timeout)
{
	int net;
	struct hostent *host;
	struct sockaddr_in sa_in;

	bzero((char*)&sa_in,sizeof(sa_in));

	if (port){
		sa_in.sin_port= htons((u_short)port);
	}else{
		return(UDM_NET_ERROR);
	}

	if ((sa_in.sin_addr.s_addr=inet_addr(hostname)) != INADDR_NONE){
		sa_in.sin_family=AF_INET;
	}else{
		host=gethostbyname(hostname);
		if (host){
			sa_in.sin_family=host->h_addrtype;
			memcpy(&sa_in.sin_addr, host->h_addr, (size_t)host->h_length);
		}else{
			return(UDM_NET_CANT_RESOLVE);
		}
	}
	net=socket(AF_INET, SOCK_STREAM, 0);

	if(connect(net, (struct sockaddr *)&sa_in, sizeof (sa_in)))
		return(UDM_NET_CANT_CONNECT);

	return(net);
}

static int UdmSearchdConnect(UDM_AGENT * query,UDM_DB *cl){
	int res=UDM_OK;
	int port=UDM_SEARCHD_PORT;
	char host[128];
	char * cport;
	
	snprintf(host,sizeof(host)-1,"%s",cl->DBADDR?cl->DBADDR:"localhost");
	if((cport=strchr(host,':'))){
		*cport='\0';
		port=atoi(cport+1);
	}
	cl->searchd=open_host(host,port,0);
	if(cl->searchd<=0){
		cl->searchd=0;
		sprintf(query->Conf->errstr,"Can't connect to searchd at '%s:%d'",host,port);
		query->Conf->errcode=1;
		res=UDM_ERROR;
	}
	return res;
}

static void UdmSearchdJustClose(UDM_AGENT * query,UDM_DB *cl){
	closesocket(cl->searchd);
	cl->searchd=0;
}

static void UdmSearchdClose(UDM_AGENT * query,UDM_DB *cl){
	UDM_SEARCHD_PACKET_HEADER hdr;
	ssize_t nsent;
	
	if(cl->searchd>0){
		/* Send goodbye */
		hdr.cmd=UDM_SEARCHD_CMD_GOODBYE;
		hdr.len=0;
		nsent=UdmSearchdSendPacket(cl->searchd,&hdr,NULL);
		closesocket(cl->searchd);
		cl->searchd=0;
	}
}

static int UdmResAddDocInfoSearchd(UDM_AGENT * query,UDM_DB *cl,UDM_RESULT * Res,size_t clnum){
	UDM_SEARCHD_PACKET_HEADER hdr;
	char * msg=NULL;
	size_t i,num=0,curnum=0;
	int * url_id, done=0;
	ssize_t nsent,nrecv;
	char * dinfo=NULL;
	
	/* Count my own documents, skip others */
	for(i=0;i<Res->num_rows;i++){
		if(clnum==UdmVarListFindInt(&Res->Doc[i].Sections,"sdnum",0))
			num++;
	}
	if(!num)return(UDM_OK);
	
		
	url_id=(int*)malloc(num*sizeof(*url_id));
	for(i=0;i<Res->num_rows;i++){
		if(clnum==UdmVarListFindInt(&Res->Doc[i].Sections,"sdnum",0)){
			url_id[curnum]=UdmVarListFindInt(&Res->Doc[i].Sections,"ID",0);
			curnum++;
		}
	}
	hdr.cmd=UDM_SEARCHD_CMD_DOCINFO;
	hdr.len=num*sizeof(*url_id);
	
	nsent=UdmSearchdSendPacket(cl->searchd,&hdr,url_id);
	
	UDM_FREE(url_id);
	
	while(!done){
		char * tok, * lt;
		nrecv=UdmRecvall(cl->searchd,&hdr,sizeof(hdr));
		
		if(nrecv!=sizeof(hdr)){
			UdmSearchdJustClose(query,cl);
			sprintf(query->Conf->errstr,"Received incomplete header from searchd (%d bytes)",(int)nrecv);
			query->Conf->errcode=1;
			return(UDM_ERROR);
		}else{
#ifdef DEBUG_SDP
			fprintf(stderr, "Received header cmd=%d len=%d\n",hdr.cmd,hdr.len);
#endif
		}
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_ERROR:
				msg=(char*)malloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
				sprintf(query->Conf->errstr,"Searchd error: '%s'",msg);
				query->Conf->errcode=1;
				free(msg);
				done=1;
				break;
			case UDM_SEARCHD_CMD_MESSAGE:
				msg=(char*)malloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Message from searchd: '%s'\n",msg);
#endif
				free(msg);
				break;
			case UDM_SEARCHD_CMD_DOCINFO:
				dinfo=(char*)malloc(hdr.len+1);
				nrecv=UdmRecvall(cl->searchd,dinfo,hdr.len);
				dinfo[hdr.len]='\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Received DOCINFO size=%d buf=%s\n",hdr.len,dinfo);
#endif				
				tok=strtok_r(dinfo,"\r\n",&lt);
				
				while(tok){
					UDM_DOCUMENT Doc;
					
					UdmDocInit(&Doc);
					UdmDocFromTextBuf(&Doc,tok);
					
					for(i=0;i<Res->num_rows;i++){
						int	Res_Doc_url_id=UdmVarListFindInt(&Res->Doc[i].Sections,"ID",0);
						int	Doc_url_id=UdmVarListFindInt(&Doc.Sections,"ID",0);
						
						if(Res_Doc_url_id!=Doc_url_id)
							continue;
						UdmDocFromTextBuf(&Res->Doc[i],tok);
						break;
					}
					tok=strtok_r(NULL,"\r\n",&lt);
					UdmDocFree(&Doc);
				}
				free(dinfo);
				done=1;
				break;
			default:
				sprintf(query->Conf->errstr,"Unknown searchd response: cmd=%d len=%d",hdr.cmd,hdr.len);
				query->Conf->errcode=1;
				done=1;
				break;
		}
	}
	return UDM_OK;
}

static int UdmSearchdSendWordRequest(UDM_AGENT * query,UDM_DB *cl,const char * q){
	UDM_SEARCHD_PACKET_HEADER hdr;
	ssize_t	nsent;
	
	hdr.cmd=UDM_SEARCHD_CMD_WORDS;
	hdr.len=strlen(q);
	nsent=UdmSearchdSendPacket(cl->searchd,&hdr,q);
	
	return UDM_OK;
}


static UDM_URL_CRD *UdmSearchdGetWordResponse(UDM_AGENT *query,UDM_RESULT *Res,UDM_DB *cl){
	UDM_URL_CRD *wrd=NULL;
	UDM_SEARCHD_PACKET_HEADER hdr;
	ssize_t	nrecv;
	char	*msg;
	int	done=0;
	char *wbuf, *p;
	UDM_WIDEWORDLIST *wwl;
	UDM_WIDEWORD *ww;
	int i;
	
	Res->total_found=0;
	
	while(!done){
		nrecv=UdmRecvall(cl->searchd,&hdr,sizeof(hdr));
		if(nrecv!=sizeof(hdr)){
			UdmSearchdJustClose(query,cl);
			sprintf(query->Conf->errstr,"Received incomplete header from searchd (%d bytes)",(int)nrecv);
			query->Conf->errcode=1;
			return(NULL);
		}
#ifdef DEBUG_SDP
		fprintf(stderr, "Received header cmd=%d len=%d\n",hdr.cmd,hdr.len);
#endif
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_ERROR:
				msg=(char*)malloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
				sprintf(query->Conf->errstr,"Searchd error: '%s'",msg);
				query->Conf->errcode=1;
				free(msg);
				done=1;
				break;
			case UDM_SEARCHD_CMD_MESSAGE:
				msg=(char*)malloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Message from searchd: '%s'\n",msg);
#endif
				free(msg);
				break;
			case UDM_SEARCHD_CMD_WORDS:
				UDM_FREE(wrd);
				wrd=(UDM_URL_CRD*)malloc(hdr.len + 1); /* FIXME: add checking */
				nrecv=UdmRecvall(cl->searchd,wrd,hdr.len);
				Res->total_found=hdr.len/sizeof(*wrd);
#ifdef DEBUG_SDP
				fprintf(stderr, "Received words size=%d nwrd=%d\n",hdr.len,Res->total_found);
#endif
				done=1;
				break;
		        case UDM_SEARCHD_CMD_WITHOFFSET:
				Res->offset = 1;
				break;
		        case UDM_SEARCHD_CMD_WWL:
			        if ((wbuf = p = (char *)UdmXmalloc(hdr.len)) != NULL) 
				  if (UdmRecvall(cl->searchd, wbuf, hdr.len))  {
				    wwl = (UDM_WIDEWORDLIST *)p;
				    p += sizeof(UDM_WIDEWORDLIST);
#ifdef DEBUG_SDP
				    fprintf(stderr, "wbuf :%x, wwl: %x, p: %x hdr.len:%d\n", wbuf, wwl, p, hdr.len);
				    fprintf(stderr, "Received WWL nwords=%d nuniq=%d\n", wwl->nwords, wwl->nuniq);
#endif
				    for(i = 0; i < wwl->nwords; i++) {
				      ww = (UDM_WIDEWORD *)p;
				      p += sizeof(UDM_WIDEWORD);
				      ww->word = p;
#ifdef DEBUG_SDP
				      fprintf(stderr, "Word: %s\n", ww->word);
#endif
				      p += ww->len + 1;
				      p += sizeof(int) - ((int)p % sizeof(int));
				      ww->uword = (int*)p;
				      p += sizeof(int) * (ww->len + 1);
				      UdmWideWordListAdd(&Res->WWList, ww, 0);
				    }
				    Res->WWList.nuniq = wwl->nuniq;
				    UDM_FREE(wbuf);
				  }
				break;
			default:
				sprintf(query->Conf->errstr,"Unknown searchd response: cmd=%d len=%d",hdr.cmd,hdr.len);
				query->Conf->errcode=1;
				done=1;
				break;
		}
	}
	return wrd;
}

int UdmSearchdFind(UDM_AGENT *query,UDM_RESULT *Res){
	UDM_URL_CRD	*wrdX[100];
	UDM_URL_CRD	*wrd,*curwrd;
	UDM_DBLIST	*sdcl=&query->Conf->sdcl;
	UDM_DB		*searchd=sdcl->db;
	size_t		i,res=UDM_OK;
	size_t		nwrd=0;
	size_t		nwrdX[100];
	int		page_number = UdmVarListFindInt(&query->Conf->Vars,"np",0);
	int		page_size   = UdmVarListFindInt(&query->Conf->Vars,"ps",20);
	char		*request;
	size_t		maxlen=1024;
#ifdef DEBUG_SEARCH
	unsigned long	debug_search_ticks;
#endif
	
	if(!sdcl->nitems){
		query->Conf->errcode=1;
		sprintf(query->Conf->errstr,"Driver is set to 'searchd' but no one SearchdAddr command is specified");
		return UDM_ERROR;
	}
	
	if (NULL==(request=(char*)malloc(maxlen))) {
		sprintf(query->Conf->errstr,"Can't allocate memory");
		query->Conf->errcode=1;
		return UDM_ERROR;
	}
	
	snprintf(request, maxlen, "%s&BrowserCharset=%s&IP=%s",
		UdmVarListFindStr(&query->Conf->Vars, "QUERY_STRING", ""),
		UdmVarListFindStr(&query->Conf->Vars, "BrowserCharset", "iso-8859-1"),
		UdmVarListFindStr(&query->Conf->Vars, "IP", ""));
	request[maxlen-1]='\0';
	for(i=0;i<sdcl->nitems;i++){
		if(UDM_OK!=(res=UdmSearchdConnect(query,&searchd[i]))){
			return res;
		}
		if(UDM_OK!=(res=UdmSearchdSendWordRequest(query,&searchd[i],request))){
			return res;
		}
	}
	free(request);
	
	for(i=0;i<sdcl->nitems;i++){
		wrdX[i]=UdmSearchdGetWordResponse(query,Res,&searchd[i]);
#ifdef DEBUG_SDP
		fprintf(stderr, "WRD[%d] total_found=%d\n",i,Res->total_found);
#endif
		nwrdX[i]=Res->total_found;
		nwrd+=Res->total_found;
	}
	
	curwrd=wrd=(UDM_URL_CRD*)malloc(sizeof(*wrd)*nwrd);
	for(i=0;i<sdcl->nitems;i++){
		if(wrdX[i]){
			size_t j;
			/* Set machine number */
			for(j=0;j<nwrdX[i];j++){
				wrdX[i][j].coord=(wrdX[i][j].coord&0xFFFFFF00)+i;
			}
			memcpy(curwrd,wrdX[i],sizeof(*curwrd)*nwrdX[i]);
			curwrd+=nwrdX[i];
			free(wrdX[i]);
		}
	}
	if (Res->offset) UdmSortSearchWordsByWeight(wrd,nwrd);
	
	Res->total_found = Res->CoordList.ncoords = nwrd;
	UDM_FREE(Res->CoordList.Coords);
	Res->CoordList.Coords = wrd;
	Res->num_rows=Res->CoordList.ncoords;
	
	Res->first=page_number*page_size;	
	if(Res->first >= Res->total_found) Res->first = (Res->total_found)? (Res->total_found - 1) : 0;
	if((Res->first+page_size)>Res->total_found){
		Res->num_rows=Res->total_found-Res->first;
	}else{
		Res->num_rows=page_size;
	}
	Res->last=Res->first+Res->num_rows-1;
	
#ifdef DEBUG_SEARCH
	fprintf(stderr,"First: %d Found: %d\n", Res->first, Res->total_found);
#endif
	
	
#ifdef DEBUG_SEARCH
	fprintf(stderr,"Start URL stuff\n");
	debug_search_ticks=UdmStartTimer();
#endif
	
	/* Allocate an array for documents information */
	Res->Doc=(UDM_DOCUMENT*)malloc(sizeof(UDM_DOCUMENT)*(Res->num_rows));
	
	/* Copy url_id and coord to result */
	for(i=0;i<Res->num_rows;i++){
		int	score=wrd[i+Res->first*Res->offset].coord;
		UdmDocInit(&Res->Doc[i]);
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"ID",wrd[i+Res->first*Res->offset].url_id);
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"Score",score);
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"Order",(int)(i + Res->first + 1));
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"sdnum",score&0xFF);
	}
	
	for(i=0;i<sdcl->nitems;i++){
		if(UDM_OK!=UdmResAddDocInfoSearchd(query,&searchd[i],Res,i));
		UdmSearchdClose(query,&searchd[i]);
	}
	
#ifdef DEBUG_SEARCH
	debug_search_ticks=UdmStartTimer()-debug_search_ticks;
	fprintf(stderr,"Stop URL stuff:\t\t%.2f\n",(float)debug_search_ticks/1000);
#endif
	
	/* first and last begins from 0, make it begin from 1 */
	Res->first++;
	Res->last++;
	return UDM_OK;
}
